/*************************************************************************
 * The contents of this file are subject to the MYRICOM MYRINET          *
 * EXPRESS (MX) NETWORKING SOFTWARE AND DOCUMENTATION LICENSE (the       *
 * "License"); User may not use this file except in compliance with the  *
 * License.  The full text of the License can found in LICENSE.TXT       *
 *                                                                       *
 * Software distributed under the License is distributed on an "AS IS"   *
 * basis, WITHOUT WARRANTY OF ANY KIND, either express or implied.  See  *
 * the License for the specific language governing rights and            *
 * limitations under the License.                                        *
 *                                                                       *
 * Copyright 2003 - 2004 by Myricom, Inc.  All rights reserved.          *
 *************************************************************************/

static const char __idstring[] = "@(#)$Id: mx__debug_dump.c,v 1.61 2006/12/14 17:05:43 loic Exp $";

/*
   Contains debugging functions to dump the state of the lib.
*/

#include "mx_auto_config.h"
#include "myriexpress.h"
#include "mx__lib_types.h"
#  include "mx__partner.h"
#include "mx__endpoint.h"
#include "mx__shim.h"
#include "mx__debug_dump.h"
#include "mx_byteswap.h"
#include "mx__handle_map.h"
#include "mx__lib.h"
#include "mx__request.h"

#if ! MX_OS_WINNT && ! defined MX_KERNEL
#include <signal.h>

static void (*old_signal)(int);
static int signal_installed;

static void
handle_signal(int sig)
{
  struct mx_endpoint *ep;
  mx_printf("Signal %d received\n", sig);

  for (ep = Mx_endpoints; ep; ep = ep->next) {
    mx_printf("Endpoint %d\n", ep->myself->eid);
    mx__dump_endpoint(ep, 0);
  }
  for (ep = Mx_endpoints; ep; ep = ep->next) {
    mx_printf("Endpoint %d/Request checks\n", ep->myself->eid);
    mx__conservation_of_matter(ep);
  }
}


#endif

static const char *rtype_names[] = {
  "send_tiny",
  "send_small",
  "send_medium",
  "send_large",
  "recv",
  "recv_large",
  "recv_self",
  "recv_shm",
  "connect",
  "connect_reply",
  "liback",
};

static const char *rstate_names[] = {
  "send_queued",
  "pending",
  "buffered",
  "completed",
  "dead",
  "acked",
  "mcp_posted",
  "replied",
  "recv_matched",
  "internal",
  "freed",
  NULL
};

void
mx__print_partner(struct mx__partner *partner)
{
  char s[18];
  char hostname[MX_MAX_HOSTNAME_LEN];
  mx_printf("%s", mx__nic_id_to_str(s, partner->nic_id, 18));
  hostname[0] = 0;
  mx_nic_id_to_hostname(partner->nic_id, hostname);
  mx_printf(" (%s)", hostname);
}

void
mx__dump_request(struct mx_endpoint *ep, union mx_request *r)
{
  int i;

  /* type */
  mx_printf("\ttype (%d): %s\n", r->basic.type, rtype_names[r->basic.type]);

  /* state */
  mx_printf("\tstate (0x%x):", r->basic.state);
  for (i=0;rstate_names[i];i++)
    if (r->basic.state & (1<< i))
      mx_printf(" %s",rstate_names[i]);
  mx_printf("\n");

  /* requeued/timeout */
#if !MX_KERNEL_LIB
  mx_printf("\trequeued: %d (timeout=%dms)\n", 
	    r->basic.requeued, 
	    (unsigned)(r->basic.timeout / mx_jiffies_hz(ep) * 1000));
#else
  mx_printf("\trequeued: %d (timeout=%"PRIx64")\n", 
	    r->basic.requeued, r->basic.timeout);
#endif

  /* partner and seqnum */
  if (r->basic.partner) {
    struct mx__partner *partner = r->basic.partner;
    mx_printf("\tdest: ");
    mx__print_partner(partner);
    mx_printf("\n");
    mx_printf("\tpartner: peer_index=%d, endpoint=%d, seqnum=0x%x\n",
	      (int)ntohs(partner->peer_index_n), partner->eid, r->basic.send_seq);
#if 0
    mx_printf("\t\tsend_acked=0x%x, recv_acked=0x%x\n"
	      "\t\trecv_seq=0x%x, fully_recv_seq=0x%x\n",
	      partner->send_acked, partner->recv_acked,
	      partner->recv_seq, partner->fully_recv_seq);
#endif
  }

  /* mcp handle */
  if (r->basic.state & MX__REQUEST_STATE_MCP)
    mx_printf("\tmcp_handle: %d\n", r->basic.mcp_handle);

  /* type specific stuff */
  switch (r->basic.type) {
  case MX__REQUEST_TYPE_RECV_LARGE:
    mx_printf("\tnotifying: %d\n", r->recv.notifying);
    /* fallthrough */
  case MX__REQUEST_TYPE_RECV:
  case MX__REQUEST_TYPE_RECV_SHM:
  case MX__REQUEST_TYPE_RECV_SELF:
    if (r->recv.unexpected) {
      mx_printf("\tunexpected\n");
    } else {
      mx_printf("\tmatch_mask: 0x%08x_%08x\n",
		MX_U32(r->recv.match_mask), MX_L32(r->recv.match_mask));
      mx_printf("\tmatch_val: 0x%08x_%08x\n",
		MX_U32(r->recv.match_info), MX_L32(r->recv.match_info));
    }
    mx_printf("\tmatched_val: 0x%08x_%08x\n",
	      MX_U32(r->recv.basic.status.match_info), MX_L32(r->recv.basic.status.match_info));
    mx_printf("\tslength=%d, rlength=%d, accum_len=%d\n",
	      r->recv.basic.status.msg_length, r->recv.r_length, r->recv.accum);
#ifndef MX_KERNEL
    mx_printf("\tseg: %p,%d\n", r->recv.segments[0].segment_ptr, r->recv.segments[0].segment_length);
#endif
    if (r->basic.type == MX__REQUEST_TYPE_RECV_LARGE) {
      mx_printf("\tlocal_rdma_id: %x\n", r->recv.local_rdma_id);
      mx_printf("\tremote_rdma: %x\n", r->recv.remote_rdma);
    }
    break;

  case MX__REQUEST_TYPE_SEND_TINY:
  case MX__REQUEST_TYPE_SEND_SMALL:
  case MX__REQUEST_TYPE_SEND_MEDIUM:
  case MX__REQUEST_TYPE_SEND_LARGE:
    mx_printf("\tmatched_val: 0x%08x_%08x\n",
	      MX_U32(r->send.basic.status.match_info), MX_L32(r->send.basic.status.match_info));
    mx_printf("\tslength=%d, xfer_length=%d\n",
	      r->send.basic.status.msg_length, r->send.basic.status.xfer_length);
#ifndef MX_KERNEL
    mx_printf("\tseg: %p,%d\n", r->send.segments[0].segment_ptr, r->send.segments[0].segment_length);
#endif
    if (r->basic.type == MX__REQUEST_TYPE_SEND_LARGE) {
      mx_printf("\tlocal_rdma_id: %x\n", r->recv.local_rdma_id);
    }
    mx_printf("\tcaller: %p\n", r->send.caller);
    break;

  case MX__REQUEST_TYPE_CONNECT:
  case MX__REQUEST_TYPE_CONNECT_REPLY:
    mx_printf("\tconnect_seq: 0x%x\n", r->connect.connect_seqnum_n);
    break;

  case MX__REQUEST_TYPE_LIBACK:
    /* FIXME: print some stuff */
    break;

  default:
    mx_printf("\tunknown request type:%d\n", r->basic.type);
  }
  mx_printf("\n");
}

static void
mx__dump_request_list(struct mx_endpoint *ep, struct mx__request_queue_head *list,
		      const char *list_name, int fulldump)
{
  struct mx__request_queue_head *elt;
  int i;
  mx_printf("%s: %d items\n", list_name, mx__count_requests(list));
  for (i = 0, elt = list->next;
       elt != list && (fulldump || i < 6);
       i++, elt = elt->next) {
    mx__dump_request(ep, MX__REQUEST_FROM_QUEUE_ELT(elt));
  }
}

void
mx__dump_endpoint(struct mx_endpoint *ep, int fulldump)
{
  unsigned int i;
  mx_printf("Dumping MX status for endpoint %d on board %d:\n",
	    ep->myself->eid, ep->board_num);
  mx_printf("handle_map: total=%d,free=%d,first=%d\n",
	    ep->handle_map->total_count,
	    ep->handle_map->free_count,
	    ep->handle_map->first_free);
  mx_printf("eventq: count=%d,index=%d,flow=%d\n", ep->event_count, ep->eventq_index, ep->eventq_flow);
  mx_printf("rdmas: max=%d, busy=%d\n", ep->rdmas.max, ep->rdmas.count);
  mx_printf("req-lookaside: free=%d,busy=%d,allocated=%d\n", ep->req_lookaside.count,
	    ep->req_lookaside.alloc_count - ep->req_lookaside.count,
	    ep->req_lookaside.alloc_count);
  for (i=0;i<ep->max_peers * ep->max_endpoints;i++) {
    struct mx__partner *p = ep->remote_ep[i];
    int app_rank = ep->application_rank_info ? ep->application_rank_info[i] : -1;
    if (p && (p->send_seq || p->recv_seq || app_rank != -1)) {
      struct mx__early_queue_head *elt;
      struct mx__early * early;
      int count = 0;
      mx_printf("peer %d: app_rank=%d, send_seq =0x%x, recv_seq = 0x%x, fully_recv = 0x%x\n",
	      i, app_rank, p->send_seq, p->recv_seq, p->fully_recv_seq);
      mx_printf("\t send_acked=0x%x, recv_acked=0x%x, send_acknum=0x%x, recv_acknum=0x%x\n",
		p->send_acked, p->recv_acked, p->send_acknum, p->recv_acknum);
      MX__FOREACH_PARTNER_EARLY(early, elt, p) {
	mx_printf("early seq=%d, length = %d, type = %d \n",
		  early->msg_seq, early->data_length, early->type);
	if (count++ >= 20 && !fulldump)
	  break;
      }
    }
  }
  mx_printf("Send-shm-pending:%d\n", ep->sendshm_count);
  mx_printf("Send-self-pending:%d\n", ep->sendself_count);
  mx_printf("Connect-replied:%d\n", ep->connect_count);
  mx__dump_request_list(ep, &ep->send_reqq, "Send", fulldump);
  for(i = 0; i < ep->ctxid_max; i++)
    mx__dump_request_list(ep, &ep->ctxid[i].recv_reqq, "Recv", fulldump);
  for(i = 0; i < ep->ctxid_max; i++)
    mx__dump_request_list(ep, &ep->ctxid[i].doneq, "Done", fulldump);
  for(i = 0; i < ep->ctxid_max; i++)
    mx__dump_request_list(ep, &ep->ctxid[i].unexpq, "Unexp", fulldump);
  mx__dump_request_list(ep, &ep->buffered_sendq, "Buffered send", fulldump);
  mx__dump_request_list(ep, &ep->mcp_connectq, "MCP Connect", fulldump);
  mx__dump_request_list(ep, &ep->mcp_connect_replyq, "MCP Connect replies", fulldump);
  mx__dump_request_list(ep, &ep->multifrag_recvq, "Multifraq recvs", fulldump);
  mx__dump_request_list(ep, &ep->large_sendq, "Large send", fulldump);
  mx__dump_request_list(ep, &ep->notifying_large_sendq, "Not notified large send", fulldump);
  mx__dump_request_list(ep, &ep->large_getq, "Large gets", fulldump);
  mx__dump_request_list(ep, &ep->resend_list, "Resends", fulldump);
  mx__dump_endpoint_stats(ep);
}


void
mx__set_app_rank(struct mx_endpoint *ep, mx_endpoint_addr_t *addr, int n)
{
  struct mx__partner * partner;
  if (!ep->application_rank_info) {
    int nbytes = sizeof(ep->application_rank_info[0])*ep->max_endpoints * ep->max_peers;
    ep->application_rank_info = mx_malloc(nbytes);
    if (!ep->application_rank_info) {
      mx_printf("Warning: mx_set_app_rank failed: NO MEM\n");
      return;
    }
    memset(ep->application_rank_info, -1, nbytes);
  }
  partner = mx__partner_from_addr(addr);
  ep->application_rank_info[ntohs(partner->peer_index_n) * ep->max_endpoints + partner->eid] = n;
}


void
mx__init_signal_handling(void)
{
#if ! MX_OS_WINNT && ! defined MX_KERNEL
  if (mx__opt.sigusr1_handler) {
    old_signal = signal(SIGUSR1,handle_signal);
    signal_installed = 1;
  }
#endif
}


void
mx__finalize_signal_handling(void)
{
#if ! MX_OS_WINNT && ! defined MX_KERNEL
  if (signal_installed) {
    signal(SIGUSR1,old_signal);
  }
#endif
}
